## -----------------------------------------------------------------------------
#| label: pre-setup
#| include: false
# Had issues on Windows (R reported that RcppTskit is not installed, which is odd).
# This tries to "restore" the library path.
lib_env <- Sys.getenv("R_LIBS")
if (nzchar(lib_env)) {
  .libPaths(unique(c(strsplit(lib_env, .Platform$path.sep)[[1]], .libPaths())))
}


## -----------------------------------------------------------------------------
#| label: setup
# install.packages("RcppTskit")

test <- require(RcppTskit)
if (!test) {
  message("RcppTskit not available; skipping vignette execution.")
  knitr::opts_chunk$set(eval = FALSE)
}


## -----------------------------------------------------------------------------
#| label: use_case_1
# Load a tree sequence
ts_file <- system.file("examples/test.trees", package = "RcppTskit")
ts <- ts_load(ts_file)
methods::is(ts)

# Print the summary of the tree sequence
ts$print()
# ts # the same as above

ts$num_individuals()

# Access the table collection
tc <- ts$dump_tables()
tc$print()

# Convert the table collection to tree sequence
ts2 <- tc$tree_sequence()

# Explore the help pages
help(package = "RcppTskit")


## -----------------------------------------------------------------------------
#| label: use_case_2
# Tree sequence in R
ts_file <- system.file("examples/test.trees", package = "RcppTskit")
ts <- ts_load(ts_file)

# If you now want to use the tskit Python API via reticulate, use
tskit <- get_tskit_py()
if (check_tskit_py(tskit)) {
  ts_py <- ts$r_to_py()
  # ... continue in reticulate Python ...
  ts_py$num_individuals # 8
  ts2_py = ts_py$simplify(samples = c(0L, 1L, 2L, 3L))
  ts2_py$num_individuals # 2
  ts2_py$num_nodes # 8
  ts2_py$tables$nodes$time # 0.0 ... 5.0093910
  # ... and to bring it back to R, use ...
  ts2 <- ts_py_to_r(ts2_py)
  ts2$num_individuals() # 2
}

# If you prefer standard (non-reticulate) Python, use
ts_file <- tempfile()
print(ts_file)
ts$dump(file = ts_file)
# ... continue in standard Python ...
# import tskit
# ts = tskit.load("insert_ts_file_path_here")
# ts.num_individuals # 8
# ts2 = ts.simplify(samples = [0, 1, 2, 3])
# ts2.num_individuals # 2
# ts2.dump("insert_ts_file_path_here")
# ... and to bring it back to R, use ...
ts2 <- ts_load(ts_file)
ts$num_individuals() # 2 (if you have run the above Python code)
file.remove(ts_file)

# You can work similarly with table collection between R & Python
tc <- ts$dump_tables()
if (check_tskit_py(tskit)) {
  tc_py <- tc$r_to_py()
  # ... continue in reticulate Python ...
  tmp <- tc_py$simplify(samples = c(0L, 1L, 2L, 3L))
  tmp
  tc_py$individuals$num_rows # 2
  tc_py$nodes$num_rows # 8
  tc_py$nodes$time # 0.0 ... 5.0093910
  # ... and to bring it back to R, use ...
  tc2 <- tc_py_to_r(tc_py)
  tc2$print()
}


## -----------------------------------------------------------------------------
#| label: use_case_3
# Write a C++ function as a multi-line character string
codeString <- '
  #include <RcppTskit.hpp>
  int ts_num_individuals(SEXP ts) {
      RcppTskit_treeseq_xptr ts_xptr(ts);
    return (int) tsk_treeseq_get_num_individuals(ts_xptr);
  }'

# Compile the C++ function
ts_num_individuals2 <- Rcpp::cppFunction(
  code = codeString,
  depends = "RcppTskit",
  plugins = "RcppTskit"
)
# We must specify both the `depends` and `plugins` arguments!

# Load a tree sequence
ts_file <- system.file("examples/test.trees", package = "RcppTskit")
ts <- ts_load(ts_file)

# Apply the compiled function
# (on the pointer)
ts_num_individuals2(ts$pointer)

# An identical RcppTskit implementation
# (available as the method of the TreeSequence class)
ts$num_individuals()


## -----------------------------------------------------------------------------
#| label: use_case_4
#| eval: false
# # Install and load the demo package
# remotes::install_github("HighlanderLab/RcppTskitTestLinking")
# library(RcppTskitTestLinking)
# 
# # Check the demo function
# print(ts_num_individuals_ptr2)
# 
# # Example tree sequence
# ts_file <- system.file("examples", "test.trees", package = "RcppTskit")
# ts <- ts_load(ts_file)
# 
# # Function from RcppTskit (working with TreeSequence R6 class)
# ts$num_individuals()
# 
# # Function from RcppTskitTestLinking (working with externalptr)
# ts_num_individuals_ptr2(ts$pointer)


## -----------------------------------------------------------------------------
#| label: session-info
sessionInfo()

